﻿using Newtonsoft.Json;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Net;
using System.Security.Cryptography;
using System.Text;
/* 示例
var _restapi = new Restapi("用户名", "密码");
_restapi.Date = DateTime.Now;
_restapi.Url = "http://fss-my.vhostgo.com/xxx/222.jpg";
_restapi.Method = "HEAD";
_restapi.CheckAction = 1;
var ret = _restapi.GetFileInfo();
*/
class Restapi
{
    private readonly int CHUNK_SIZE = 1024 * 1024;
    /// <summary>
    /// 用户名
    /// </summary>
    public string Username { get; set; }
    /// <summary>
    /// 密码
    /// </summary>
    public string Password { get; set; }
    /// <summary>
    /// 设置验证方式 0为基础认证，其他为签名认证
    /// </summary>
    public int CheckAction { get; set; }
    /// <summary>
    /// 请求的url
    /// </summary>
    public string Url { get; set; }
    /// <summary>
    /// 请求方法
    /// </summary>
    public string Method { get; set; }
    /// <summary>
    /// 日期(基础认证可不设置)
    /// </summary>
    public DateTime Date { get; set; }
    /// <summary>
    /// 设置文件内容的md5加密
    /// </summary>
    public string ContentMd5 { get; set; }
    /// <summary>
    /// 是否异步执行，适用于删除和上传文件
    /// </summary>
    public bool Async { get; set; }
    /// <summary>
    /// 
    /// </summary>
    /// <param name="username"></param>
    /// <param name="password"></param>
    /// <param name="action"></param>
    public Restapi(string username = null, string password = null, int action = 0)
    {
        Username = username;
        Password = password;
        CheckAction = action;
    }
    /// <summary>
    /// 获取认证
    /// </summary>
    /// <returns></returns>
    public string GetAuthorization()
    {
        if (CheckAction == 0)
        {
            return "Basic " + Convert.ToBase64String(new ASCIIEncoding().GetBytes(string.Format("{0}:{1}", Username, Password)));
        }
        var urlAddress = new Uri(Url);
        var uri = urlAddress.AbsolutePath + urlAddress.Query;
        var data = string.Format("{0}&{1}&{2}", Method, uri, Date.ToUniversalTime().ToString("R"));
        if (!string.IsNullOrEmpty(ContentMd5))
        {
            data += "&" + ContentMd5;
        }
        var password = Convert.ToBase64String(new ASCIIEncoding().GetBytes(Password));
        var hmac = new HMACSHA1(Encoding.UTF8.GetBytes(password));
        byte[] buffer = Encoding.UTF8.GetBytes(data);
        var sign = Convert.ToBase64String(hmac.ComputeHash(buffer));
        return "WESTYUN " + Username + ':' + sign;
    }
    /// <summary>
    /// 获取目录文件列表
    /// </summary>
    /// <param name="iter">分页开始位置</param>
    /// <param name="limit">获取的文件数量</param>
    /// <param name="order">asc 或 desc，按文件名升序或降序排列。默认 asc</param>
    /// <returns></returns>
    public string GetList(string iter = null, int? limit = null, string order = null)
    {
        WebHeaderCollection header = new WebHeaderCollection
        {
            { "x-list-iter", iter },
            { "x-list-limit", limit.ToString() },
            { "x-list-order", order }
        };
        return GetRequestObj(header)["data"];
    }
    /// <summary>
    /// 分块获取文件内容
    /// </summary>
    /// <param name="start"></param>
    /// <param name="end"></param>
    /// <returns></returns>
    public string GetContent(int? start = null, int? end = null)
    {
        WebHeaderCollection header = new WebHeaderCollection();
        if (start != null && end == null)
        {
            header.Add("Range", "bytes=" + start + "-");
        }
        else if (start == null && end != null)
        {
            header.Add("Range", "bytes=-" + end);
        }
        else if (start != null && end != null)
        {
            header.Add("Range", "bytes=" + start + "-" + end);
        }
        return GetRequestObj(header)["data"];
    }
    /// <summary>
    /// 获取文件信息
    /// </summary>
    /// <returns></returns>
    public Dictionary<string, string> GetFileInfo()
    {
        var ret = GetRequestObj();
        return (from t in ret
                where t.Key.IndexOf("x-west-") >= 0
                select t).ToDictionary(k => k.Key, v => v.Value);
    }
    /// <summary>
    /// 复制or移动文件
    /// </summary>
    /// <param name="source">源文件地址</param>
    /// <param name="type">0-复制 其它移动</param>
    /// <param name="mkdir">制文件时目录不存在自动创建，默认false</param>
    /// <param name="directive">copy(默认),merge,replace,delete</param>
    /// <param name="meta">设置文件元信息</param>
    /// <returns></returns>
    public bool CopyOrMoveFile(string source, int type, string mkdir = null, string directive = null, Dictionary<string, string> meta = null)
    {
        try
        {
            WebHeaderCollection header = new WebHeaderCollection();
            if (type == 0)
            {
                header.Add("x-west-copy-source", source);
            }
            else
            {
                header.Add("x-west-move-source", source);
            }
            if (mkdir != null)
            {
                header.Add("x-west-automkdir", mkdir);
            }
            if (directive != null)
            {
                header.Add("x-west-metadata-directive", directive);
            }

            if (meta != null)
            {
                foreach (var k in meta.Keys)
                {
                    header.Add("x-west-meta-" + k, meta[k]);
                }
            }
            var response = GetRequestObj(header);
            return (response["code"] == "200");
        }
        catch
        {
            return false;
        }
    }
    /// <summary>
    /// 创建目录
    /// </summary>
    /// <returns></returns>
    public bool CreateFolder()
    {
        try
        {
            WebHeaderCollection header = new WebHeaderCollection {
                {"folder","true" }
            };
            var response = GetRequestObj(header);
            return (response["code"] == "200");
        }
        catch
        {
            return false;
        }
    }
    /// <summary>
    /// 删除目录或文件
    /// </summary>
    /// <returns></returns>
    public bool DeleteFolderOrFlie()
    {
        try
        {
            WebHeaderCollection header = new WebHeaderCollection();
            if (Async)
            {
                header.Add("x-west-async", "true");
            };
            var response = GetRequestObj(header);
            return (response["code"] == "200");
        }
        catch
        {
            return false;
        }
    }
    /// <summary>
    /// 设置文件的metadata
    /// </summary>
    /// <param name="metaInfo"></param>
    /// <returns></returns>
    public bool SetMetadata(Dictionary<string, string> metaInfo)
    {
        try
        {
            WebHeaderCollection header = new WebHeaderCollection();
            foreach (var k in metaInfo.Keys)
            {
                header.Add("x-west-meta-" + k, metaInfo[k]);
            }
            var response = GetRequestObj(header);
            return (response["code"] == "200");
        }
        catch
        {
            return false;
        }
    }
    /// <summary>
    /// 上传文件
    /// </summary>
    /// <param name="path">文件路径</param>
    /// <param name="type">文件类型，默认使用文件扩展名作为文件类型</param>
    /// <param name="secret">文件密钥，用于保护文件，防止文件被直接访问</param>
    /// <param name="ttl">文件元信息, 指定文件的生存时间，单位天，最大支持180天</param>
    /// <param name="thumb">图片预处理参数</param>
    /// <param name="origin">文件元信息</param>
    /// <returns></returns>
    public bool UploadFile(string path, string type = null, string secret = null, int? ttl = null, string thumb = null, Dictionary<string, string> origin = null)
    {
        try
        {
            WebHeaderCollection header = new WebHeaderCollection();
            var data = File_get_contents(path);
          
            if (Async)
            {
                header.Add("x-west-async", "true");
            }
            if (ContentMd5 != null)
            {
                header.Add("Content-MD5", ContentMd5);
            }

            if (secret != null)
            {
                header.Add("Content-Secret", secret);
            }
            if (ttl != null)
            {
                header.Add("x-west-meta-ttl", ttl.ToString());
            }
            if (thumb != null)
            {
                header.Add("x-gmkerl-thumb", thumb);
            }
            if (origin != null)
            {
                foreach (var k in origin.Keys)
                {
                    header.Add("x-west-meta-" + k, origin[k]);
                }
            }
            var response = GetRequestObj(header, data, type);

            return (response["code"] == "200");
        }
        catch
        {
            //文件是否已存在
            throw;
            return false;
        }
    }
    /// <summary>
    /// 分块上传文件
    /// </summary>
    /// <param name="path"></param>
    /// <param name="type"></param>
    /// <param name="secret"></param>
    /// <param name="ttl"></param>
    /// <param name="origin"></param>
    /// <returns></returns>
    public bool UploadBigFile(string path, string type = null, string secret = null, int? ttl = null, Dictionary<string, string> origin = null)
    {
        try
        {
            long length = new FileInfo(path).Length;
            bool initResult = BigFileInit(length, type, secret, ttl, origin);
            if (initResult == false)
            {
                throw new ArgumentException("upload init failed");
            }
            using (FileStream fsRead = new FileStream(@path, FileMode.Open, FileAccess.Read))
            {
                int i = 0; int readCount = 0;
                byte[] byts = new byte[CHUNK_SIZE];
                while ((readCount = fsRead.Read(byts, 0, CHUNK_SIZE))>0)
                {                  
                    var uploadRes = UploadChunk(i, byts);
                    Array.Clear(byts, 0, byts.Length);
                    if (!uploadRes) throw new ArgumentException("第" + i.ToString() + "块分片上传失败");
                    i++;                   
                }
                return BigFileComplete();
            }
        }
        catch
        {
            return false;
        }
    }
    private bool BigFileInit(long length, string type = null, string secret = null, int? ttl = null, Dictionary<string, string> origin = null)
    {
        try
        {
            WebHeaderCollection header = new WebHeaderCollection {
                {"x-west-multi-disorder","true" },{ "x-west-multi-stage","initiate"},{ "x-west-multi-length", length.ToString()}
            };
            if (type != null)
            {
                header.Add("x-west-multi-type", type);
            }
            if (secret != null)
            {
                header.Add("Content-Secret", secret);
            }
            if (ttl != null)
            {
                header.Add("x-west-meta-ttl", ttl.ToString());
            }
            if (origin != null)
            {
                foreach (var k in origin.Keys)
                {
                    header.Add("x-west-meta-" + k, origin[k]);
                }
            }
            var response = GetRequestObj(header);
            header.Clear();
            return (response["code"] == "200");
        }
        catch
        {
            return false;
        }
    }
    /// <summary>
    /// 分块上传
    /// </summary>
    /// <param name="chunkId"></param>
    /// <param name="data"></param>
    /// <returns></returns>
    private bool UploadChunk(int chunkId, byte[] data)
    {
        try
        {
            WebHeaderCollection header = new WebHeaderCollection { { "x-west-multi-disorder", "true" }, { "x-west-multi-stage", "upload" }, { "x-west-part-id", chunkId.ToString() } };
            var response = GetRequestObj(header, data);
            return (response["code"] == "200");
        }
        catch { return false; }
    }
    /// <summary>
    /// 上传完成
    /// </summary>
    /// <returns></returns>
    private bool BigFileComplete()
    {
        WebHeaderCollection header = new WebHeaderCollection { { "x-west-multi-disorder", "true" }, { "x-west-multi-stage", "complete" } };
        var response = GetRequestObj(header);
        return (response["code"] == "200");
    }
    private byte[] File_get_contents(string fileName)
    {
        try
        {
            byte[] sContents;
            if (fileName.ToLower().IndexOf("http:") > -1)
            {
                WebClient wc = new WebClient();
                sContents = wc.DownloadData(fileName);
               // sContents = Encoding.UTF8.GetString(response);//以utf8的编码格式返回，如果返回值有乱码应该是跟页面的编码格式不同，自行更换编码格式
            }
            else
            {
                using (FileStream fs = new FileStream(fileName, FileMode.Open, FileAccess.Read))
                {
                
                        sContents = new byte[fs.Length];
                        fs.Read(sContents, 0, (int)fs.Length);
               
                }
            }
            return sContents;
        }
        catch (Exception ex)
        {
            throw ex;
        }
    }
    /// <summary>
    /// url编码(暂不用)
    /// </summary>
    /// <param name="str"></param>
    /// <returns></returns>
    public static string UrlEncode(string str)
    {
        return str;
        /*
        Regex reg = new Regex(@"^(?<url>https?\://[^/]+?)/(?<key1>.*?)(\?(?<key2>.*?))?$");
        var mc = reg.Match(str);
        var url =mc.Groups["url"].Value;
        var key1 = System.Web.HttpUtility.UrlEncode(mc.Groups["key1"].Value);
        var key2 = System.Web.HttpUtility.UrlEncode(mc.Groups["key2"].Value);
        var ret = url+"/" + key1;
        if (key2 != "") ret += "?" + key2;
        return ret;
        */
    }
    private Dictionary<string, string> GetRequestObj(WebHeaderCollection hearder = null, byte[] putdata = null, string type = null)
    {
        string authorization = GetAuthorization();
        HttpWebRequest request = (HttpWebRequest)WebRequest.Create(Url);
        request.KeepAlive = false;
        request.ServicePoint.Expect100Continue = false;
        if (hearder != null)
        {
            request.Headers = hearder;
        }
        if (type != null)
        {
            request.ContentType = type;
        }
        request.Headers.Add("Authorization", authorization);
        if (Date != null) request.Date = Date;
        request.Method = Method;
        if (putdata != null)
        {
            byte[] b = putdata;// Encoding.UTF8.GetBytes(putdata);
            request.ContentLength = b.Length;
            using (Stream stream = request.GetRequestStream())
            {
                stream.Write(b, 0, b.Length);
            }
        }
        try
        {
            using (var response = (HttpWebResponse)request.GetResponse())
            {
                using (var reader = new StreamReader(response.GetResponseStream(), Encoding.UTF8))
                {
                    string ret = reader.ReadToEnd();
                    if (ret.IndexOf("\"code\":") < 0)
                    {
                        var dicret = new Dictionary<string, string>();
                        if (Method == "HEAD")
                        {
                            for (int i = 0; i < response.Headers.Keys.Count; i++)
                            {
                                dicret.Add(response.Headers.Keys[i], response.Headers.Get(i));
                            }
                            return dicret;
                        }
                        dicret.Add("data", ret);
                        return dicret;
                    }
                    return JsonConvert.DeserializeObject<Dictionary<string, string>>(ret);

                }
            }
        }
        catch (WebException wex)
        {
            WebResponse wr = wex.Response;
            using (Stream st = wr.GetResponseStream())
            {
                using (StreamReader sr = new StreamReader(st, Encoding.UTF8))
                {
                    var ret = sr.ReadToEnd();
                    if (ret == "") throw;
                    return JsonConvert.DeserializeObject<Dictionary<string, string>>(ret);
                }
            }
        }
        catch
        {
            throw;
        }
        finally
        {
            if (request != null)
            {
                request.Abort();
            }
        }
    }

}
